<?php

/**
 * This is the model class for table "Lekcja".
 *
 * The followings are the available columns in table 'Lekcja':
 * @property double $id
 * @property string $scenariusz
 * @property double $order
 */
class Lekcja extends CActiveRecord
{
	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Lekcja the static model class
	 */

	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Lekcja';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('order', 'numerical'),
			array('scenariusz', 'length', 'max'=>4000),
			array('order, scenariusz', 'required'),

			array('id, scenariusz, order', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'turas'=>array(self::HAS_MANY, 'Tura', 'idLekcji'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Lekcja', 'ID'),
			'scenariusz'=>Yii::t('Lekcja', 'Scenariusz'),
			'order'=>Yii::t('Lekcja', 'Kolejność'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id);
		$criteria->compare('scenariusz', $this->scenariusz, true);
		$criteria->compare('order', $this->order);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * To zapytanie pobiera lekcje z rozbiciem po 5 dla każdej klasy nauczyciela i statusem
	 * 0 - można rozpocząć - jeżeli nie trwa w tym momencie tura oraz ta lekcja jest pierwszą niewykonaną
	 * 1 - nie można rozpocząć - ta tura nie jest pierwszą nierozpoczętą
	 * 2 - runda trwa
	 * 3 - runa zakończona w przeszłości
	 *
	 * @param type $adminId
	 * @return \CSqlDataProvider
	 */
	public function getAsTables($adminId) {
		$sql = 'SELECT t6."id", t5."idKlasy", t6."scenariusz", "czasRozpoczecia", t8."nazwa", ROWNUM as "rownum", t6."order",
			CASE WHEN
				t6."id" IN (SELECT t12."id" FROM "Lekcja" t12 WHERE t12."order" = (
					SELECT MIN(t10."order") FROM "Lekcja" t10 WHERE t10."order" >= 0 AND t10."id" NOT IN (
						SELECT t9."idLekcji" FROM "Tura" t9 WHERE t9."idKlasy" = t8."id") AND (
							SELECT rownum FROM "Tura" t13 WHERE t13."idKlasy" = t8."id"  AND t13."czasRozpoczecia" <= SYSDATE AND t13."czasZakonczenia" >= SYSDATE
						) IS NULL
					)
				)
				THEN 0 /* round start allowed */
			WHEN  "czasRozpoczecia" IS NULL AND "czasZakonczenia" IS NULL
				THEN 1 /* round start not allowed */
			WHEN  "czasRozpoczecia" IS NOT NULL AND "czasZakonczenia" IS NOT NULL AND "czasZakonczenia" <= SYSDATE
				THEN 2 /* round lasts */
			WHEN  "czasRozpoczecia" IS NOT NULL AND "czasZakonczenia" IS NOT NULL AND "czasZakonczenia" > SYSDATE
				THEN 3 /* round is over */  END as "status",
			("czasZakonczenia" - sysdate) * 86400 "duration"
		FROM "AdminKlasy" t5
		JOIN "Lekcja" t6 ON t6."id" IN (SELECT t."id"
		FROM "Lekcja" t
		WHERE t."order" >= (SELECT CASE WHEN MAX(t4."order") IS NULL THEN 1 ELSE MAX(t4."order") END
			FROM "Tura" t3
			LEFT JOIN "Lekcja" t4 ON (t4."id" = t3."idLekcji")
			WHERE (t3."idKlasy" = t5."idKlasy" AND t3."czasRozpoczecia" <= SYSDATE)) AND rownum <=5)
		LEFT JOIN "Tura" t7 ON (t7."idLekcji" = t6."id" AND t7."idKlasy" = t5."idKlasy")
		LEFT JOIN "Klasa" t8 ON t8."id" = t5."idKlasy"
		WHERE t5."idAdmina" = :adminId AND (t8."czyTestowa" = 0 OR t8."czyTestowa" = 1 AND t8."id" = (SELECT MAX("id") FROM "Klasa" LEFT JOIN "AdminKlasy" ON "AdminKlasy"."idKlasy" = "Klasa"."id" WHERE "AdminKlasy"."idAdmina" = :adminId AND "Klasa"."czyTestowa" = 1))
		ORDER BY t5."idKlasy", t6."order"';

		return new CSqlDataProvider($sql, array(
			'pagination'=>false,
			'params'=>array(
				':adminId' => $adminId
			)
		));
	}

	/**
	 * To zapytanie pobiera lekcje dla danej klasy wraz ze statusem
	 * 0 - można rozpocząć - jeżeli nie trwa w tym momencie tura oraz ta lekcja jest pierwszą niewykonaną
	 * 1 - nie można rozpocząć - ta tura nie jest pierwszą nierozpoczętą
	 * 2 - runda trwa
	 * 3 - runa zakończona w przeszłości
	 *
	 * @param type $classId
	 * @return \CSqlDataProvider
	 */
	public function getByClass($classId) {
		/*
		 */
		$sql = 'SELECT t."id", ROWNUM as "rownum", t."scenariusz", t."order",
			CASE WHEN
				t."id" IN (SELECT t0."id" FROM "Lekcja" t0 WHERE t0."order" = (
					SELECT MIN(t3."order") FROM "Lekcja" t3 WHERE t3."order" >= 0 AND t3."id" NOT IN (
						SELECT t5."idLekcji" FROM "Tura" t5 WHERE t5."idKlasy" = :classId) AND (
							SELECT rownum FROM "Tura" t6 WHERE t6."idKlasy" = :classId  AND t6."czasRozpoczecia" <= SYSDATE AND t6."czasZakonczenia" >= SYSDATE
						) IS NULL
					)
				)
				THEN 0 /* round start allowed */
			WHEN "czasRozpoczecia" IS NULL AND "czasZakonczenia" IS NULL
				THEN 1 /* round start not allowed */
			WHEN  "czasRozpoczecia" IS NOT NULL AND "czasZakonczenia" IS NOT NULL AND "czasZakonczenia" <= SYSDATE
				THEN 2 /* round is over */
			WHEN  "czasRozpoczecia" IS NOT NULL AND "czasZakonczenia" IS NOT NULL AND "czasZakonczenia" > SYSDATE
				THEN 3 /* round is lasts */
			END as "status",
			("czasZakonczenia" - sysdate) * 86400 "duration"
		FROM "Lekcja" t
		LEFT JOIN "Tura" t2 ON (t."id" = t2."idLekcji" AND t2."idKlasy" = :classId)
		WHERE t."order" >= 0 AND t2."idKlasy" IS NULL OR t2."idKlasy" = :classId
		ORDER BY "order"';

		return new CSqlDataProvider($sql, array(
			'pagination'=>false,
			'params' => array(
				':classId' => $classId,
			)
		));
	}

	/**
	 * To zapytanie sprawdza czy daną lekcję można rozpocząć
	 * Warunki rozpoczęcia:
	 * 1. Nie trwa dla tej klasy inna lekcja
	 * 2. Ta lekcja jest pierwsza w kolejności nierozpoczętych
	 * 3. Klasa należy do danego nauczyciela
	 * 4. Lekcja nie jest zakończona
	 * Wywoływane przy rozpoczęciu lekcji dla bezpieczeństwa
	 *
	 * @param type $teacherId
	 * @param type $lessonId
	 * @param type $classId
	 * @return type
	 */
	public function startAllowed($teacherId, $lessonId, $classId)
	{
		/*
		 */
		/*/* pobiera turę trwającą */
		$sql = '
		SELECT CASE WHEN
				/* ilość tur trwających = 0 */
				(SELECT COUNT(*) FROM
				"Tura" t WHERE
				(t."czasRozpoczecia" <= SYSDATE AND t."czasZakonczenia" >= SYSDATE) AND t."idKlasy"=:classId) = 0
			AND
				/* lekcja nie jest przeszla */
				 :lessonId NOT IN
				  (SELECT t1."idLekcji" FROM
				  "Tura" t1 WHERE
				  (t1."czasZakonczenia" <= SYSDATE) AND t1."idKlasy"=:classId)
			AND
				/* id lekcji jest pierwsza mozliwa lekcja */
				:lessonId = (SELECT t4."id" FROM "Lekcja" t4 WHERE t4."order" = (
					SELECT MIN(t3."order") FROM "Lekcja" t3 WHERE t3."order" >= 0 AND t3."id" NOT IN (
						SELECT t2."idLekcji" FROM "Tura" t2 WHERE "idKlasy" = :classId)
					)
				)
			AND
				/* obecny nauczyciel jest nauczycielem tej klasy */
				(SELECT COUNT(*) FROM "AdminKlasy" t5 WHERE t5."idAdmina" = :teacherId AND t5."idKlasy" = :classId ) > 0
			AND
				/* order lekcji <= ILOSC_AKTYWNYCH_LEKCJI */
				(SELECT COUNT(*) FROM "Lekcja" t6 WHERE t6."id" = :lessonId AND t6."order" <= :maxLesson) > 0
			THEN
				1
			ELSE
				0
			END AS "lessonStartAllowed"
		FROM DUAL';

		return Yii::app()->db->createCommand($sql)->queryScalar(array(
			':teacherId'=>$teacherId,
			':lessonId'=>$lessonId,
			':classId'=>$classId,
			':maxLesson'=>ParametryGry::getValue('ILOSC_AKTYWNYCH_LEKCJI')
		));
	}
}
